//	TorusGames2DMouse.c
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include <math.h>


//	How close must two clicks be to count as a double-click in the Apples2D game?
#define DOUBLE_CLICK_RADIUS		0.01

//	On iOS, a scroll coasts to a gradual stop
//	after the user has lifted his/her finger.
#define COASTING_DECELERATION	2.0	//	distance / sec²
#define MAX_COASTING_DISTANCE	4.0	//	distance
#define MAX_COASTING_SPEED		( sqrt(2.0 * COASTING_DECELERATION * MAX_COASTING_DISTANCE) )	//	distance / sec


static void	ScrollTheBoard2D(ModelData *md, double aDeltaH, double aDeltaV);


#ifdef __APPLE__
#pragma mark -
#pragma mark 2D mouse or touch
#endif

void MouseDown2D(	//	also works as "TouchBegan2D()"
	ModelData	*md,
	double		aMouseH,				//	horizontal coordinate (intrinsic units rightward from center, in range [-0.5, +0.5])
	double		aMouseV,				//	 vertical  coordinate (intrinsic units   upward  from center, in range [-0.5, +0.5])
	bool		aScrollFlag,			//	Dragging with the shift key requests a scroll (desktop only)
	bool		aMarkFlag,				//	Clicking with the right mouse button or with the control key
										//		(desktop only in both cases)
										//		serves to mark apples in the Apples game.
	bool		aTwoFingerGestureFlag,	//	Dragging in response to a two-finger gesture
	bool		aFlickGestureFlag,		//	Dragging in response to a flick gesture (iOS only)
	bool		aTemporalDoubleClick)	//	desktop only
{
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	bool	thePreviousClickWasDoubleClickEligible;
#endif

	//	In ViewBasicSmall or ViewRepeating, rescale mouse coordinates
	//	from [-0.5, +0.5] to [-1.5, +1.5].
	switch (md->itsViewType)
	{
		case ViewBasicLarge:
			//	no change
			break;
		
		case ViewBasicSmall:
		case ViewRepeating:
			aMouseH *= 3.0;
			aMouseV *= 3.0;
			break;
	}
	
	//	Convert mouse from window coordinates to fundamental domain coordinates.
	aMouseH -= md->itsOffset.itsH;
	aMouseV -= md->itsOffset.itsV;
	if (md->itsOffset.itsFlip)
		aMouseH = -aMouseH;

	//	Gesture handling
	//
	//	Handle two-finger and flick gestures independently
	//	of whatever may be going on with the hand cursor,
	//	to avoid any possibility of conflict when iOS
	//	reports UIGestureRecognizerStateBegan and then immediately
	//	*after* that calls -touchesCancelled:withEvent: .
	//	Note that even though the iOS version of Torus Games
	//	never explicitly draws the hand cursor, it's still
	//	useful to think about what the "invisible hand cursor"
	//	is doing, and to never let two-finger and flick gestures
	//	interact with it.
	
	if (aTwoFingerGestureFlag
	 || aFlickGestureFlag)
	{
		//	No need for coasting while the gesture is still in progress.
		
		md->itsCoastingStatus = CoastingNone;
		
		md->its2DCoastingVelocity[0] = 0.0;
		md->its2DCoastingVelocity[1] = 0.0;
	}
	else	//	Standard touch or mouse handling
	{
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE

		//	Note whether thePreviousClickWasDoubleClickEligible, for use this time.
		thePreviousClickWasDoubleClickEligible = md->its2DPreviousClickWasDoubleClickEligible;
		
		//	Set its2DPreviousClickWasDoubleClickEligible for use next time.
		switch (md->its2DHandStatus)
		{
			case HandNone:
				//	If the user clicks once to enter Torus Cursor Mode,
				//	and then immediately clicks a second time to make a move in a game,
				//	that should *not* count as a double click to exit Torus Cursor Mode.
				md->its2DPreviousClickWasDoubleClickEligible = false;
				break;
			
			case HandFree:
				//	Accept a double click to exit Torus Cursor Mode.
				md->its2DPreviousClickWasDoubleClickEligible = true;
				break;
			
			case HandScroll:
			case HandDrag:
				//	Clicks can't arrive while scrolling or dragging,
				//	except in unusual circumstances (for example,
				//	if the user is using a trackpad and a mouse simultaneously).
				md->its2DPreviousClickWasDoubleClickEligible = false;
				break;
		}

#endif
		//	Handle the mouse-down according to its2DHandStatus.
		switch (md->its2DHandStatus)
		{
			case HandNone:
			
				//	Place our hand internal cursor where
				//		the system cursor was (desktop) or
				//		the user's finger tapped (iOS).
				md->its2DHandPlacement.itsH		= aMouseH;
				md->its2DHandPlacement.itsV		= aMouseV;
				md->its2DHandPlacement.itsFlip	= md->itsOffset.itsFlip;

				//	Normalize its2DHandPlacement to the fundamental domain.
				Normalize2DPlacement(&md->its2DHandPlacement, md->itsTopology);

				//	Note that we now have the cursor.
				md->its2DHandStatus = HandFree;

#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
				//	For Tic-Tac-Toe, Gomoku, Pool or Apples let's break here,
				//	to avoid accidentally making a tic-tac-toe or gomoku move,
				//	shooting the cue ball or uncovering an unintended apple.
				if (md->itsGame == Game2DTicTacToe
				 || md->itsGame == Game2DGomoku
				 || md->itsGame == Game2DPool
				 || md->itsGame == Game2DApples)
					break;
				//	Otherwise...
#endif
				//	Fall through to next case,
				//	in case the user is trying to click on something...
			case HandFree:

				//	Begin a drag or scroll.
				if
				(
					//	While an animation is in progress, all mouse-down events
					//	get treated as scrolls, to avoid interfering with whatever
					//	action the game has in progress.
					md->itsSimulationStatus == SimulationNone
				 &&
					//	If the user requested a scroll, honor that request.
					! aScrollFlag
				 &&
					//	Otherwise let the game decides whether the mouse hit point
					//	determines a drag (if some object is hit)
					//	or a scroll (is no object is hit).
					md->itsGame2DDragBegin != NULL
				 &&
					(*md->itsGame2DDragBegin)(md, aMarkFlag)
				)
				{
					md->its2DHandStatus = HandDrag;
				}
				else
				{
					md->its2DHandStatus = HandScroll;

					md->its2DCoastingVelocity[0] = 0.0;
					md->its2DCoastingVelocity[1] = 0.0;
				}

				break;

			default:	//	unused (I think)
				break;
		}
	
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
		//	The operating system (macOS) judges a click to be a double-click
		//	based solely on the elapsed time since the preceding click.
		//	In the case of Game2DApples we insist that the two clicks be spatially close
		//	as well, so that rapid clicking (to eat apples) does not accidentally
		//	exit the user from the torus or Klein bottle.
		md->its2DDoubleClickFlag =
		(
			thePreviousClickWasDoubleClickEligible
		 &&
			aTemporalDoubleClick
		 && 
			(	
				md->itsGame != Game2DApples
			 ||	
				Shortest2DDistance(	md->its2DHandPlacement.itsH,
									md->its2DHandPlacement.itsV,
									md->its2DPrevClickPlacement.itsH,
									md->its2DPrevClickPlacement.itsV,
									md->itsTopology,
									NULL) < DOUBLE_CLICK_RADIUS
			)
		);
		md->its2DPrevClickPlacement = md->its2DHandPlacement;
#else
		UNUSED_PARAMETER(aTemporalDoubleClick);
#endif
	}
	
	//	Ask the idle-time routine to redraw the scene.
	md->itsChangeCount++;
}

void MouseMove2D(	//	also works as "TouchMoved2D()"
	ModelData	*md,
	double		aMouseDeltaH,			//	same coordinate system as in MouseMove()
	double		aMouseDeltaV,
	double		aMouseDeltaT,
	bool		aTwoFingerGestureFlag,	//	Dragging in response to a two-finger gesture
	bool		aFlickGestureFlag)		//	Dragging in response to a flick gesture (iOS only)
{
	double	theHandLocalDeltaH,
			theHandLocalDeltaV;

#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE

	//	On a Mac or Windows computer, let a given mouse motion
	//	map to the same intrinsic displacement, regardless of whether
	//	the game is shown as ViewBasicLarge, ViewBasicSmall or ViewRepeating.

#ifdef TORUS_GAMES_FOR_TALK
	//	During a talk, drag faster to make it easier, for example,
	//	to move the flounder a long distance without lifting my finger.
	switch (md->itsViewType)
	{
		case ViewBasicLarge:
			aMouseDeltaH *= 2.0;
			aMouseDeltaV *= 2.0;
			break;
		
		case ViewBasicSmall:
		case ViewRepeating:
			aMouseDeltaH *= 2.0 * 3.0;
			aMouseDeltaV *= 2.0 * 3.0;
			break;
	}
#endif

#else	//	touch interface

	//	On a phone or tablet, in ViewBasicSmall or ViewRepeating
	//	triple the (ΔH, ΔV) to match the coordinate change
	//	from [-0.5, +0.5] to [-1.5, +1.5], so that game motions
	//	will exactly track the user's finger.
	switch (md->itsViewType)
	{
		case ViewBasicLarge:
			//	no change
			break;
		
		case ViewBasicSmall:
		case ViewRepeating:
			aMouseDeltaH *= 3.0;
			aMouseDeltaV *= 3.0;
			break;
	}

#endif	//	mouse or touch interface

	//	Interpret the mouse motion as a motion of the hand
	//	relative to its own coordinate system.
	theHandLocalDeltaH = aMouseDeltaH;
	theHandLocalDeltaV = aMouseDeltaV;

	//	See note on "Gesture handling" in MouseDown2D() above.
	
	if (aTwoFingerGestureFlag
	 || aFlickGestureFlag)
	{
		//	Even though the 3D games interpret them differently,
		//	the 2D games interpret the two-finger gesture and
		//	the flick gesture in exactly the same way:  as a scroll.

		ScrollTheBoard2D(md, theHandLocalDeltaH, theHandLocalDeltaV);

		//	Compute and remember its2DCoastingVelocity.
		if (aMouseDeltaT <= 0.0 || aMouseDeltaT > 1.0)	//	just to be safe
			aMouseDeltaT = 1.0;
		md->its2DCoastingVelocity[0] = theHandLocalDeltaH / aMouseDeltaT;
		md->its2DCoastingVelocity[1] = theHandLocalDeltaV / aMouseDeltaT;
	}
	else	//	Standard touch or mouse handling
	{
		//	Handle the MouseMove2D according to the its2DHandStatus.
		switch (md->its2DHandStatus)
		{
			case HandNone:
				break;

			case HandFree:	//	Occurs only in desktop version.

				//	Scroll the hand along, taking into account
				//	the hand's orientation within the fundamental domain.
				md->its2DHandPlacement.itsH += (md->its2DHandPlacement.itsFlip ? -theHandLocalDeltaH : theHandLocalDeltaH);
				md->its2DHandPlacement.itsV += theHandLocalDeltaV;
				Normalize2DPlacement(&md->its2DHandPlacement, md->itsTopology);

				//	Some games don't care about the position of a free hand, but some do.
				if (md->itsGame2DHandMoved != NULL)
					(*md->itsGame2DHandMoved)(md);

				break;

			case HandDrag:

				//	The game will take responsibility for modifying
				//	the motion to avoid collisions and then moving
				//	both the hand and the object as necessary.
				//	The hand is invisible in the iOS version, but gets moved anyhow.
				if (md->itsGame2DDragObject != NULL)
					(*md->itsGame2DDragObject)(md, theHandLocalDeltaH, theHandLocalDeltaV);

				break;

			case HandScroll:

				ScrollTheBoard2D(md, theHandLocalDeltaH, theHandLocalDeltaV);

				//	Compute and remember its2DCoastingVelocity.
				if (aMouseDeltaT <= 0.0 || aMouseDeltaT > 1.0)	//	just to be safe
					aMouseDeltaT = 1.0;
				md->its2DCoastingVelocity[0] = theHandLocalDeltaH / aMouseDeltaT;
				md->its2DCoastingVelocity[1] = theHandLocalDeltaV / aMouseDeltaT;

				break;
		}
	}
	
	//	Ask the idle-time routine to redraw the scene.
	md->itsChangeCount++;
}

void MouseUp2D(	//	also works as "TouchEnded2D()"
	ModelData	*md,
	double		aDragDuration,			//	in seconds
	bool		aTwoFingerGestureFlag,	//	Dragging in response to a two-finger gesture
	bool		aFlickGestureFlag,		//	Dragging in response to a flick gesture (iOS only)
	bool		aTouchSequenceWasCancelled)
{
	double	theSpeed,
			theFactor;

	//	See note on "Gesture handling" in MouseDown2D() above.
	
	if (aTwoFingerGestureFlag
	 || aFlickGestureFlag)
	{
		//	Let a scroll gradually coast to a stop.
		//	SimulationUpdate() will call CoastingMomentum() once per frame
		//	until CoastingMomentum() clears itsCoastingStatus.
		md->itsCoastingStatus = Coasting2DTranslation;
		
		//	Don't let the board coast further than MAX_COASTING_DISTANCE.
		//	A quick Physics 101 calculation shows that a board with initial velocity
		//	MAX_COASTING_SPEED = sqrt(2.0 * COASTING_DECELERATION * MAX_COASTING_DISTANCE)
		//	travels a distance MAX_COASTING_DISTANCE before coming to rest.
		theSpeed = sqrt(md->its2DCoastingVelocity[0] * md->its2DCoastingVelocity[0]
					  + md->its2DCoastingVelocity[1] * md->its2DCoastingVelocity[1]);
		if (theSpeed > MAX_COASTING_SPEED)
		{
			theFactor = MAX_COASTING_SPEED / theSpeed;
			md->its2DCoastingVelocity[0] *= theFactor;
			md->its2DCoastingVelocity[1] *= theFactor;
		}
	}
	else	//	Standard touch or mouse handling
	{
		switch (md->its2DHandStatus)
		{
			case HandNone:
			case HandFree:
				break;

			case HandScroll:

				//	Here it's tempting to write code like
				//
				//		if (aTouchSequenceWasCancelled)
				//		{
				//			md->itsCoastingStatus = CoastingNone;
				//		}
				//		else
				//		{
				//			...
				//		}
				//
				//	but alas iOS 10 first handles a flick or 2-finger gesture,
				//	and then immediately afterwards cancels the plain touch sequence
				//	(the wrong order in my opinion).  So if we set
				//	md->itsCoastingStatus = CoastingNone in response to a cancelled touch sequence,
				//	we'd be canceling the coasting that the flick or 2-finger gesture
				//	had just initiated.  Instead let's let the following lines of code
				//	redundantly set itsCoastingStatus and redundantly limit the speed.
				//
				//		Caution:  This code is fragile.  It's called after
				//		the gesture has been handled.  If you ever add or change
				//		anything here, watch out for unintented side effects.
				//

				//	Let a scroll gradually coast to a stop.
				//	SimulationUpdate() will call CoastingMomentum() once per frame
				//	until CoastingMomentum() clears itsCoastingStatus.
				md->itsCoastingStatus = Coasting2DTranslation;
				
				//	Don't let the board coast further than MAX_COASTING_DISTANCE.
				//	A quick Physics 101 calculation shows that a board with initial velocity
				//	MAX_COASTING_SPEED = sqrt(2.0 * COASTING_DECELERATION * MAX_COASTING_DISTANCE)
				//	travels a distance MAX_COASTING_DISTANCE before coming to rest.
				theSpeed = sqrt(md->its2DCoastingVelocity[0] * md->its2DCoastingVelocity[0]
							  + md->its2DCoastingVelocity[1] * md->its2DCoastingVelocity[1]);
				if (theSpeed > MAX_COASTING_SPEED)
				{
					theFactor = MAX_COASTING_SPEED / theSpeed;
					md->its2DCoastingVelocity[0] *= theFactor;
					md->its2DCoastingVelocity[1] *= theFactor;
				}

				break;

			case HandDrag:

				//	Let the game finalize the drag.
				if (md->itsGame2DDragEnd != NULL)
					(*md->itsGame2DDragEnd)(md, aDragDuration, aTouchSequenceWasCancelled);

				break;
		}

		//	Free the hand.
#ifdef TORUS_GAMES_2D_TOUCH_INTERFACE
		md->its2DHandStatus = HandNone;
#endif
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
		if (md->its2DHandStatus != HandNone)	//	Maybe user typed ESC while mouse was down?
			md->its2DHandStatus = HandFree;
#endif
	}
		
	//	Ask the idle-time routine to redraw the scene.
	md->itsChangeCount++;
}

void MouseGone2D(ModelData *md)
{
	//	Typically we expect the user to complete any scrolling or dragging
	//	before relinquishing the torus cursor, but just in case a drag is still
	//	in progress when the user hits the ESC key, or in case some other program
	//	unexpectedly grabs the cursor, let's simulate a MouseUp2D().
	//	Pass aTouchSequenceWasCancelled = true to cancel any pending action.
	MouseUp2D(	md,
				0.0,	//	value will be ignored, because we're passing aTouchSequenceWasCancelled = true
				false,
				false,
				true);

	//	We're done with the cursor.
	md->its2DHandStatus = HandNone;
	
	//	Ask the idle-time routine to redraw the scene.
	md->itsChangeCount++;
}


void MouseWheel2D(
	ModelData	*md,
	double		aDeltaH,
	double		aDeltaV)
{
	ScrollTheBoard2D(md, aDeltaH, aDeltaV);

	md->itsChangeCount++;
}


#ifdef __APPLE__
#pragma mark -
#pragma mark 2D coasting and scrolling
#endif

void CoastingMomentum2D(
	ModelData	*md,
	double		aTimeInterval)	//	in seconds
{
	double	theSpeedDecrease,
			theSpeed,
			theFactor,
			theDeltaH,
			theDeltaV;

	if (md->itsCoastingStatus == Coasting2DTranslation)	//	should never fail
	{
		//	The user has lifted his/her finger,
		//	but for a brief period we let the scroll or drag
		//	continue under its own momentum.

		theSpeedDecrease = aTimeInterval * COASTING_DECELERATION;
		
		theSpeed = sqrt(md->its2DCoastingVelocity[0] * md->its2DCoastingVelocity[0]
					  + md->its2DCoastingVelocity[1] * md->its2DCoastingVelocity[1]);
		
		if (theSpeed > theSpeedDecrease)
		{
			theFactor = (theSpeed - theSpeedDecrease) / theSpeed;
			md->its2DCoastingVelocity[0] *= theFactor;
			md->its2DCoastingVelocity[1] *= theFactor;
			
			theDeltaH = md->its2DCoastingVelocity[0] * aTimeInterval;
			theDeltaV = md->its2DCoastingVelocity[1] * aTimeInterval;

			//	its2DCoastingVelocity is relative to the view itself,
			//	so it the fundamental square is coasting northeastward,
			//	it keeps coasting northeastward, even if it wraps around
			//	and comes back flipped.
			//
			//	In a torus we could move the hand relative to the board,
			//	so that as the board coasts the hand remains still
			//	relative to the view itself.  In a Klein bottle, though,
			//	that would be impossible, because different images
			//	of the hand would need to move in different directions
			//	relative to the board;  the effect is seen most clearly
			//	in tiling mode.
			//
			ScrollTheBoard2D(md, theDeltaH, theDeltaV);
		}
		else
		{
			//	The "coasting" period is over.

			md->itsCoastingStatus			= CoastingNone;
			md->its2DCoastingVelocity[0]	= 0.0;
			md->its2DCoastingVelocity[1]	= 0.0;
		}
		
		//	Ask the idle-time routine to redraw the scene.
		md->itsChangeCount++;
	}
}


static void ScrollTheBoard2D(
	ModelData	*md,
	double		aDeltaH,
	double		aDeltaV)
{
	//	Different images of the hand may have different chiralities
	//	in the Klein bottle, especially in Tiling view.
	//	The scroll can't possibly agree with all of them.
	//	So scroll relative to an unreflected hand
	//	to keep the interface as intuitive as possible.
	md->itsOffset.itsH += aDeltaH;
	md->itsOffset.itsV += aDeltaV;
	Normalize2DOffset(&md->itsOffset, md->itsTopology);
}
